//Class which buffers file access.   Lacks the extra features provided by BufferedFile, but this is useful for
//buffering output to a text file or other file formats where the extra byte that BufferedFile sticks on the
//front of the file to indicate endianness interferes with the output file.

//PlainBufferedFile still provides all of the functionality of a BFile: It now provides Read/Write access, random
//access, etc.  All the regular BFile calls can be mixed with the PlainBufferedFile calls, although the BFile calls
//won't benefit from the buffering (they force the buffer to flush, and actually introduce inefficiency - use
//them carefully; in general try to stick to the BufferedRead and BufferedWrite calls)


#ifndef _SGB_PLAIN_BUFFERED_FILE_H_
#define _SGB_PLAIN_BUFFERED_FILE_H_


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <File.h>


//******************************************************************************************************
//**** PlainBufferedFile CLASS DECLARATION
//******************************************************************************************************
class PlainBufferedFile: public BFile
{
	public:
		//openMode may only be read or write, not read/write for all constructors
		PlainBufferedFile(const entry_ref* ref, uint32 openMode, int32 BufferSize = 4095);
		PlainBufferedFile(const BEntry* entry, uint32 openMode, int32 BufferSize = 4095);
		PlainBufferedFile(const char* path, uint32 openMode, int32 BufferSize = 4095);
		PlainBufferedFile(BDirectory* dir, const char* path, uint32 openMode, int32 BufferSize = 4095);

		virtual ~PlainBufferedFile();	//Destroys the PlainBufferedFile, but DOES NOT write any buffered data, as
									//a file error could occur without the opportunity to report the error.
									//You must call FlushBuffer() before destroying the PlainBufferedFile in
									//order to write any data in the write buffer.
		status_t FlushBuffer();	//Flushes the read buffer and moves the file pointer to the position that
								//was being read from.  It is automatically called when necessary when
								//working within the file, but if you have buffered write data, call
								//it yourself before deleting the PlainBufferedFile to make sure the data is 
								//written successfully (the destructor can't return an error code).
								//Returns B_NO_ERROR if the buffer was flushed successfully.
		status_t InitCheck();	//Returns B_NO_ERROR if the PlainBufferedFile was created successfully.

		//Reads or writes the raw data contained in ReadBuffer or WriteBuffer.  Returns the number of bytes
		//actually read or written.
		int32 BufferedRead(void* Buffer, size_t BytesToRead);
		int32 BufferedWrite(const void* Buffer, size_t BytesToWrite);

		//BFile overrides
		virtual	ssize_t Read(void *buffer, size_t size);
		virtual	ssize_t ReadAt(off_t pos, void *buffer, size_t size);
		virtual	ssize_t Write(const void *buffer, size_t size);
		virtual	ssize_t WriteAt(off_t pos, const void *buffer, size_t size);
		virtual off_t Seek(off_t position, uint32 seek_mode);
		virtual	status_t SetSize(off_t size);

		//BFile functions not accessible from BFile as virtuals
		status_t SetTo(const entry_ref *ref, uint32 open_mode);
		status_t SetTo(const BEntry *entry, uint32 open_mode);
		status_t SetTo(const char *path, uint32 open_mode);
		status_t SetTo(const BDirectory *dir, const char *path, uint32 open_mode);

	private:
		void AllocAndInitBuffer(int32 Buffer_Size);
		void CheckNewFileStatus();

		int32 BufferSize;
		char* Buffer;
		int32 BytesInBuffer;
		int32 BufferPos;
		uint32 openMode;
		status_t Valid;
		bool WriteBuffered;
};


#endif

